/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "modbusserver.h"
#include "miscellaneous.h"
#include <QDebug>
#include <QFile>
#include <QNetworkInterface>
#include <QRandomGenerator>
#include <QUrl>

//
// TODO:
// #include <QRandomGenerator> apparently provides 'round()' functionality... for some reason.
// But this seems kind of odd and we should probably just use qround instead and get rid of QRandomGenerator?
// Or alternatively just provide #include <math.h> instead?
//

ModbusServer::ModbusServer(QString modbus_file):mbServer(nullptr)             //mbServer = nullptr
{
	QFile file(modbus_file);
	if(!file.open(QIODevice::ReadOnly))
	{
		qDebug() << "Modbus File unavailable!" << file.errorString() << "\n" << file.fileName();
	}
	else
	{
		/* Read the file and save values to configmap */
		while(!file.atEnd())
		{
			QString line;
			line = file.readLine();
			line.replace("\t","");
			line.replace("\r","");
			line.replace("\n","");

			if (!(line.startsWith("#") || line.isEmpty()))
			{
				 QStringList list = line.split(QRegExp("\\s*=\\s*"));
				 if(list.count() == 2)
				 {
					QStringList list2 = list.at(1).split(",");
					double multiplier = 1.0;
					QString modbus_register = "";

					if (list2.count() >= 1)
					{
						modbus_register = list2.at(0);
					}
					if (list2.count() >= 2)
					{
						multiplier = list2.at(1).toDouble();
					}
					multiplier_map.insert(list.at(0), multiplier);
					modbus_map.insert(list.at(0), modbus_register);
				 }
				 else
				 {
					qDebug() << "line syntax incorrect: " << list << "ignored line";
				 }
			}
		}
	}
	file.close();
}

ModbusServer::~ModbusServer()
{
	if (mbServer)
	{
		mbServer->disconnectDevice();
	}
	delete mbServer;
}

void ModbusServer::Initialize()
{
	mbServer = new QModbusTcpServer(this);
	lastServer = new QModbusTcpServer(this);

	//Primary Modbus Map
	QModbusDataUnitMap reg;
//	reg.insert(QModbusDataUnit::Coils, { QModbusDataUnit::Coils, 0, 100 });
//	reg.insert(QModbusDataUnit::DiscreteInputs, { QModbusDataUnit::DiscreteInputs, 0, 100 });
	reg.insert(QModbusDataUnit::InputRegisters, { QModbusDataUnit::InputRegisters, 8192, 10000 });
	reg.insert(QModbusDataUnit::HoldingRegisters, { QModbusDataUnit::HoldingRegisters, 24576, 25000 });
	mbServer->setMap(reg);

	//Secondary modbus map for making comparisons to prevent duplicate inputs
	QModbusDataUnitMap last_regs;
//	last_regs.insert(QModbusDataUnit::Coils, { QModbusDataUnit::Coils, 0, 100});
//	last_regs.insert(QModbusDataUnit::DiscreteInputs, { QModbusDataUnit::DiscreteInputs, 0, 100 });
//	last_regs.insert(QModbusDataUnit::InputRegisters, { QModbusDataUnit::InputRegisters, 8192, 10000});
	last_regs.insert(QModbusDataUnit::HoldingRegisters, { QModbusDataUnit::HoldingRegisters, 24576, 25000});
	lastServer->setMap(last_regs);

	connect(mbServer, &QModbusServer::dataWritten,
			this, &ModbusServer::Holding_Register_Update_Handler);
	connect(mbServer, &QModbusServer::stateChanged,
			this, &ModbusServer::onConnectedStateChanged);
	connect(mbServer, &QModbusServer::errorOccurred,
			this, &ModbusServer::handleDeviceError);
}

//
//TODO:
//Some better Modbus Error handling would be nice I guess...
//
void ModbusServer::handleDeviceError(QModbusDevice::Error newError)
{
	if (newError == QModbusDevice::NoError || !mbServer)
	{
		return;
	}

	qDebug() << mbServer->errorString();
}

void ModbusServer::onConnectedStateChanged(int state)
{
	QString address =  mbServer->connectionParameter(QModbusDevice::NetworkAddressParameter).toString();
	int port = mbServer->connectionParameter(QModbusDevice::NetworkPortParameter).toInt();
	QString error = mbServer->errorString();
	if (state == QModbusDevice::UnconnectedState)
	{
		emit signal_modbus_connected_enable(false, address, port, error);
	}
	else if (state == QModbusDevice::ConnectedState)
	{
		emit signal_modbus_connected_enable(true, address, port, error);
	}
}

void ModbusServer::set_listen_enable(bool enable)
{
	if (enable)
	{
		QString mbAddress;

		//if address is defined in modbus.txt use that one, otherwise use a valid IPv4 address or localhost address as a last resort.
		if (!modbus_map.contains("modbus_slave_address"))
		{
			QList<QHostAddress> ipAddressesList = QNetworkInterface::allAddresses();
			for (int i = 0; i < ipAddressesList.size(); ++i)
			{
				/* use the first non-localhost IPv4 address */
				if (ipAddressesList.at(i) != QHostAddress::LocalHost && ipAddressesList.at(i).toIPv4Address())
				{
					mbAddress = ipAddressesList.at(i).toString();
					break;
				}
			}
			if (mbAddress.isEmpty())	// if we did not find one, use IPv4 localhost
			{
				mbAddress = QHostAddress(QHostAddress::LocalHost).toString();
			}
		}
		else
		{
			mbAddress = modbus_map.value("modbus_slave_address");
		}

		const QUrl url = QUrl::fromUserInput(mbAddress + ":" + modbus_map.value("modbus_slave_port"));
		mbServer->setConnectionParameter(QModbusDevice::NetworkAddressParameter, url.host());
		mbServer->setConnectionParameter(QModbusDevice::NetworkPortParameter, url.port());
		mbServer->setServerAddress(modbus_map.value("modbus_slave_ID").toInt());

		if (!mbServer->connectDevice())
		{
			qDebug() << "Connect failed: " + mbServer->errorString();
		}
	}
	else
	{
		mbServer->disconnectDevice();
	}
}

/**********************************************************************************************************************************************************************************
 * REGISTERS SET/GETS
 * *******************************************************************************************************************************************************************************/
//Set Register
void ModbusServer::setRegister(QModbusDataUnit::RegisterType reg_type, quint16 address, quint16 value)
{
	if (!mbServer)
		return;

	bool ok = true;
	ok = mbServer->setData(reg_type, address, value);

	if (!ok)
	{
		//
		// TODO:
		// It would be really nice if the error handling here was a bit nicer.
		// Like knowing which register could not be written, instead of just seeing address '0'.
		// But that would require passing the modbus_map key string, rather than the value string
		// Also this function should return a bool.
		//
		qDebug() << "Modbus::setRegister - could not set register type" << reg_type << "at address:" << QString::number(address) << "to value:" << value << "Error:" << mbServer->errorString();
	}
}

void ModbusServer::setInputRegister(quint16 address, quint16 value)
{
	setRegister(QModbusDataUnit::InputRegisters, address, value);
}

void ModbusServer::setHoldingRegister(quint16 address, quint16 value)
{
	setRegister(QModbusDataUnit::HoldingRegisters, address, value);
}

//Get Register
quint16 ModbusServer::getRegister(QModbusDataUnit::RegisterType reg_type, quint16 address)
{
	if (!mbServer)
		return -1;

	quint16 reg_value = 0;
	if (!mbServer->data(reg_type, address, &reg_value))
	{
		qDebug() << "getRegister: failed to get data |" + mbServer->errorString();
	}

	return (reg_value);
}

quint16 ModbusServer::getInputRegister(quint16 address)
{
	return getRegister(QModbusDataUnit::InputRegisters, address);
}

quint16 ModbusServer::getHoldingRegister(quint16 address)
{
	return getRegister(QModbusDataUnit::HoldingRegisters, address);
}


//Set Register Bit
void ModbusServer::setBit(quint16 &value, quint16 bit_pos, bool enable)
{
	if (enable == true)
	{
		value |= (1 << bit_pos);	//Overwrite to 1;
	}
	else
	{
		value &= ~(1 << bit_pos);	//Overwrite to 0;
	}
}

void ModbusServer::setRegisterBit(QModbusDataUnit::RegisterType reg_type, quint16 address, quint16 bit_pos, bool enable)
{
	quint16 reg_value= 0;
	if (mbServer->data(reg_type, address, &reg_value))
	{
		setBit(reg_value, bit_pos, enable);
		setRegister(reg_type, address, reg_value);
	}
}

void ModbusServer::setInputRegisterBit(quint16 address, quint16 bit_pos, bool enable)
{
	setRegisterBit(QModbusDataUnit::InputRegisters, address, bit_pos, enable);
}

void ModbusServer::setHoldingRegisterBit(quint16 address, quint16 bit_pos, bool enable)
{
	setRegisterBit(QModbusDataUnit::HoldingRegisters, address, bit_pos, enable);
}

//Get Register Bit
bool ModbusServer::getRegisterBit(QModbusDataUnit::RegisterType reg_type, quint16 address, quint16 bit_pos)
{
	quint16 reg_value= 0;
	if (!mbServer->data(reg_type, address, &reg_value))
	{
		qDebug() << "getRegisterBit: failed to get data  |" + mbServer->errorString();
	}

	return (reg_value & (1<<bit_pos));
}

bool ModbusServer::getInputRegisterBit(quint16 address, quint16 bit_pos)
{
	return getRegisterBit(QModbusDataUnit::InputRegisters, address, bit_pos);
}

bool ModbusServer::getHoldingRegisterBit(quint16 address, quint16 bit_pos)
{
	return getRegisterBit(QModbusDataUnit::HoldingRegisters, address, bit_pos);
}



/**********************************************************************************************************************************************************************************
 * INPUT REGISTERS
 * *******************************************************************************************************************************************************************************/

//
// TODO:
#warning There should probably be way more rounding actions happening in this part.
//

void ModbusServer::handler_generator_ready_get(bool enable)
{
	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 0, enable);
}

void ModbusServer::handler_RF_enable_get(int intrasys_num, bool enable)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 1, enable);
}

void ModbusServer::handler_DLL_enable_get(int intrasys_num, bool enable)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 4, enable);
}

void ModbusServer::handler_PWM_settings_get(int intrasys_num, double frequency_hz, int duty_cycle)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	bool enable;

	if (duty_cycle >= 1 && duty_cycle <= 99)	//PWM enabled <> External Triggering Mode disabled.
	{
		enable = true;
	}
	else
	{
		enable = false;
	}

	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 5, enable);
	setInputRegister(modbus_map.value("IR_PWM_frequency").toUShort(), frequency_hz * multiplier_map.value("IR_PWM_frequency"));
	setInputRegister(modbus_map.value("IR_PWM_duty_cycle").toUShort(), duty_cycle);
}

void ModbusServer::handler_power_get(int intrasys_num, double power_dbm, double power_watt)
{
	(void)intrasys_num, (void)power_dbm; 	//unused parameter - cast to void to prevent compiler warning.

	power_watt = round(power_watt * multiplier_map.value("IR_FWD_power_setpoint"));		//Handle the rounding properly
	setInputRegister(modbus_map.value("IR_FWD_power_setpoint").toUShort(), (quint16)power_watt);
}

void ModbusServer::handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11_loss, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_ratio)
{
	//unused parameters - cast to void to prevent compiler warning.
	(void)intrasys_num, (void)PA_power_fwd_dbm, (void)PA_power_rfl_dbm, void(PA_s11_ratio);

	/* S11 should be negative; Cap off S11 values to 0 Maximum.
	 * Also make the negative S11 value absolute (force positive), because registers only support positive values. */
	double s11 = PA_s11_loss;
	if (s11 > 0)
	{
		s11 = 0;
	}

	setInputRegister(modbus_map.value("IR_FWD_power").toUShort(), (quint16)round(PA_power_fwd_watt * multiplier_map.value("IR_FWD_power")));
	setInputRegister(modbus_map.value("IR_RFL_power").toUShort(), (quint16)round(PA_power_rfl_watt * multiplier_map.value("IR_RFL_power")));
	setInputRegister(modbus_map.value("IR_return_loss").toUShort(), (quint16)round(abs(s11) * multiplier_map.value("IR_return_loss")));
}

void ModbusServer::handler_frequency_get(int intrasys_num, double frequency_mhz)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	// TODO: I'd kind of prefer if we split large values in high + low registers, so that this becomes re-useable in the future.
	quint16 frequency = round(frequency_mhz * multiplier_map.value("IR_actual_frequency"));
	setInputRegister(modbus_map.value("IR_actual_frequency").toUShort(), frequency);
}

void ModbusServer::handler_get_phase(int intrasys_num, double phase_degrees)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	quint16 phase = round(phase_degrees * multiplier_map.value("IR_phase"));
	setInputRegister(modbus_map.value("IR_phase").toUShort(), phase);
}

void ModbusServer::handler_get_clock_source(int intrasys_num, int clock_source)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	setInputRegister(modbus_map.value("IR_clock_source").toUShort(), clock_source);
}

void ModbusServer::handler_SWP_measurement_get(int intrasys_num, QString SWP_raw_data)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	if(SWP_raw_data.contains("$SWPD,") && SWP_raw_data.contains("OK\r\n"))
	{
		QStringList SWP_data = SWP_raw_data.split("\r\n");

		//Remove the OK entry the empty one that comes after.
		QVector<double>	SWP_freq_data;
		QVector<double>	SWP_frw_data;
		QVector<double>	SWP_rfl_data;
		QVector<double>	SWP_s11_dbm_data;
//		QVector<double>	SWP_s11_watt_data;

		SWP_freq_data.resize(SWP_data.count()-2);
		SWP_frw_data.resize(SWP_data.count()-2);
		SWP_rfl_data.resize(SWP_data.count()-2);
		SWP_s11_dbm_data.resize(SWP_data.count()-2);
//		SWP_s11_watt_data.resize(SWP_data.count()-2);

		QRegExp regexp("\\$SWP\\D?,\\d+,(\\d+.?\\d+),(\\d+.?\\d+),(\\d+.?\\d+)");

		for (int i = 0; i < SWP_data.count() - 2; i++)
		{
			 if (regexp.indexIn(SWP_data.at(i))>=0)
			 {
				  QString string1 = regexp.cap(1);
				  QString string2 = regexp.cap(2);
				  QString string3 = regexp.cap(3);

				  SWP_freq_data[i] = string1.toDouble();
				  SWP_frw_data[i] = string2.toDouble();
				  SWP_rfl_data[i] = string3.toDouble();

				  SWP_s11_dbm_data[i] = SWP_rfl_data[i] - SWP_frw_data[i];
//				  SWP_s11_watt_data[i] = convert_dbm_to_watt(SWP_rfl_data[i]) / convert_dbm_to_watt(SWP_frw_data[i]);

				  /* Perhaps theoretically a division by zero is possible here, but the minimum power value that can be set is 0.1W / 20dBm anyway...
				   * Since we get our power values in dBm and convert to watt, there's no such thing as 0 dbm anyway... therefor in practice divide by zero doesn't occur. */
			 }
		}

		double frequency_best_match = 0;
		double s11_best_match = __INT_MAX__;


		for (int i = 0; i < SWP_s11_dbm_data.size(); i++)
		{
			if (s11_best_match > SWP_s11_dbm_data.at(i))
			{
				s11_best_match = SWP_s11_dbm_data.at(i);		//finds the best S11 match (lowest S11 in dB)
				frequency_best_match = SWP_freq_data.at(i);		//set frequency to frequency of best S11 match
			}
		}

		frequency_best_match = round(frequency_best_match * multiplier_map.value("IR_S11_frequency_best_match"));

		//S11 should be negative; Cap off S11 values to 0 Maximum;
		//Make the negative S11 value absolute (force positive), because registers only support positive values.
		if (s11_best_match > 0)
			s11_best_match = 0;
		s11_best_match = round(abs(s11_best_match) * multiplier_map.value("IR_S11_loss_best_match"));

		double s11_start = SWP_s11_dbm_data.at(0);
		if (s11_start > 0)
			s11_start = 0;
		s11_start = round((abs(s11_start) * multiplier_map.value("IR_S11_loss_start_freq")));

		double s11_stop = SWP_s11_dbm_data.at(SWP_s11_dbm_data.size()-1);
		if (s11_stop > 0)
			s11_stop = 0;
		s11_stop = round((abs(s11_stop) * multiplier_map.value("IR_S11_loss_stop_freq")));

		//Write Sweep data.
		setInputRegister(modbus_map.value("IR_S11_frequency_best_match").toUShort(), frequency_best_match);
		setInputRegister(modbus_map.value("IR_S11_loss_best_match").toUShort(), s11_best_match);
		setInputRegister(modbus_map.value("IR_S11_loss_start_freq").toUShort(), s11_start);
		setInputRegister(modbus_map.value("IR_S11_loss_stop_freq").toUShort(), s11_stop);

//		//Sweep Complete True
//		setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 3, true);
	}
//	else
//	{
//		//Sweep Complete false
//		setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 3, false);
//	}

	//Sweep Running False;
//	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 2, false);
}

void ModbusServer::handler_SWP_settings_get(int intrasys_num, double frequency_start, double frequency_stop, double frequency_step, double power_dbm, double power_watt)
{
	(void)intrasys_num, (void)power_dbm;	//unused parameters - cast to void to prevent compiler warning.

	double frq_start = frequency_start * multiplier_map.value("IR_SWP_start_frequency");
	double frq_stop = frequency_stop * multiplier_map.value("IR_SWP_stop_frequency");
	double frq_step = frequency_step * multiplier_map.value("IR_SWP_step_frequency");
	double pwr_watt = round(power_watt * multiplier_map.value("IR_SWP_power"));

	setInputRegister(modbus_map.value("IR_SWP_start_frequency").toUShort(), frq_start);
	setInputRegister(modbus_map.value("IR_SWP_stop_frequency").toUShort(), frq_stop);
	setInputRegister(modbus_map.value("IR_SWP_step_frequency").toUShort(), frq_step);
	setInputRegister(modbus_map.value("IR_SWP_power").toUShort(), pwr_watt);
}

void ModbusServer::handler_temperature_get(int intrasys_num, double temperature)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	temperature = round(temperature * multiplier_map.value("IR_temperature"));
	setInputRegister(modbus_map.value("IR_temperature").toUShort(), (quint16)temperature);
}

void ModbusServer::handler_error_get(int intrasys_num, quint64 error, QStringList error_messages)
{
	(void)intrasys_num, (void) error_messages;	//unused parameters - cast to void to prevent compiler warning.

	quint16 error_P1 = 0;	//Register #1 -> 1~16 bits
	quint16 error_P2 = 0;	//Register #2 -> 17~32 bits
	quint16 error_P3 = 0;	//Register #3 -> 33~48 bits
	quint16 error_P4 = 0;	//Register #4 -> 49~64 bits

	/* The '1' must be cast to a 64bit unsigned int / unsigned long long
	 * Otherwise the bitwise AND-operation goes wrong */
	for (int i = 0; i <= 15; i++)
	{
		setBit(error_P1, i, error & ((quint64) 1<<i));
	}
	for (int i = 16; i <= 31; i++)
	{
		setBit(error_P2, i-16, error & ((quint64) 1<<i));
	}
	for (int i = 32; i <= 47; i++)
	{
		setBit(error_P3, i-32, error & ((quint64) 1<<i));
	}
	for (int i = 48; i <= 63; i++)
	{
		setBit(error_P4, i-48, error & ((quint64) 1<<i));
	}

	setInputRegister(modbus_map.value("IR_SG_error_P4").toUShort(), error_P4);
	setInputRegister(modbus_map.value("IR_SG_error_P3").toUShort(), error_P3);
	setInputRegister(modbus_map.value("IR_SG_error_P2").toUShort(), error_P2);
	setInputRegister(modbus_map.value("IR_SG_error_P1").toUShort(), error_P1);
}

void ModbusServer::handler_datalogging_enable_get(bool enable)
{
	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 7, enable);
}

void ModbusServer::handler_datalogging_storage_sufficient(bool valid)
{
	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 13, valid);
}

void ModbusServer::handler_PSU_enable_combined_get(int intrasys_num, bool enable)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 2, enable);
}

void ModbusServer::handler_PSU_IU_get(int intrasys_num, int psu_num, double voltage, double current, double power)
{
	(void)intrasys_num;	//unused parameter - cast to void to prevent compiler warning.

	switch (psu_num)
	{
	case 0:
		setInputRegister(modbus_map.value("IR_PSU1_voltage").toUShort(), round(voltage * multiplier_map.value("IR_PSU1_voltage")));
		setInputRegister(modbus_map.value("IR_PSU1_current").toUShort(), round(current * multiplier_map.value("IR_PSU1_current")));
		setInputRegister(modbus_map.value("IR_PSU1_power").toUShort(), round(power * multiplier_map.value("IR_PSU1_power")));
		break;
	case 1:
		setInputRegister(modbus_map.value("IR_PSU2_voltage").toUShort(), round(voltage * multiplier_map.value("IR_PSU2_voltage")));
		setInputRegister(modbus_map.value("IR_PSU2_current").toUShort(), round(current * multiplier_map.value("IR_PSU2_current")));
		setInputRegister(modbus_map.value("IR_PSU2_power").toUShort(), round(power * multiplier_map.value("IR_PSU2_power")));
		break;
	case 2:
		setInputRegister(modbus_map.value("IR_PSU3_voltage").toUShort(), round(voltage * multiplier_map.value("IR_PSU3_voltage")));
		setInputRegister(modbus_map.value("IR_PSU3_current").toUShort(), round(current * multiplier_map.value("IR_PSU3_current")));
		setInputRegister(modbus_map.value("IR_PSU3_power").toUShort(), round(power * multiplier_map.value("IR_PSU3_power")));
		break;
	case 3:
		setInputRegister(modbus_map.value("IR_PSU4_voltage").toUShort(), round(voltage * multiplier_map.value("IR_PSU4_voltage")));
		setInputRegister(modbus_map.value("IR_PSU4_current").toUShort(), round(current * multiplier_map.value("IR_PSU4_current")));
		setInputRegister(modbus_map.value("IR_PSU4_power").toUShort(), round(power * multiplier_map.value("IR_PSU4_power")));
		break;
	default:
		break;
	}
}

void ModbusServer::handler_PSU_power_efficiency_get(int intrasys_num, double efficiency)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	efficiency = round(efficiency * multiplier_map.value("IR_PSU_power_efficiency"));
	setInputRegister(modbus_map.value("IR_PSU_power_efficiency").toUShort(), efficiency);
}

void ModbusServer::handler_PSU_dissipation_get(int intrasys_num, double val)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	double dissipation = val;
	if (val < 0)
	{
		dissipation = 0;
	}
	dissipation = round(dissipation * multiplier_map.value("IR_PSU_power_dissipation"));
	setInputRegister(modbus_map.value("IR_PSU_power_dissipation").toUShort(), dissipation);
}

void ModbusServer::handler_SGx_communication_working_get(bool state)
{
	setInputRegisterBit(modbus_map.value("IR_status").toUShort(),11, state);
}

void ModbusServer::handler_PSU_communication_working_get(bool state)
{
	setInputRegisterBit(modbus_map.value("IR_status").toUShort(),12, state);
}

/* Get Phase values from different channels of a Phase Gain Board */
void ModbusServer::handler_get_PGB_phase(int intrasys_num, int PGB_channel_num, double phase)
{
	(void)intrasys_num;		//unused parameter - cast to void to prevent compiler warning.

	switch (PGB_channel_num)
	{
	case 1:
		setInputRegister(modbus_map.value("IR_PGB_ch1_phase").toUShort(), round(phase * multiplier_map.value("IR_PGB_ch1_phase")));
		break;
	case 2:
		setInputRegister(modbus_map.value("IR_PGB_ch2_phase").toUShort(), round(phase * multiplier_map.value("IR_PGB_ch2_phase")));
		break;
	case 3:
		setInputRegister(modbus_map.value("IR_PGB_ch3_phase").toUShort(), round(phase * multiplier_map.value("IR_PGB_ch3_phase")));
		break;
	case 4:
		setInputRegister(modbus_map.value("IR_PGB_ch4_phase").toUShort(), round(phase * multiplier_map.value("IR_PGB_ch4_phase")));
		break;
	default:
		break;
	}
}


/**********************************************************************************************************************************************************************************
 * HOLDING REGISTERS
 * *******************************************************************************************************************************************************************************/
void ModbusServer::Holding_Register_Update_Handler(QModbusDataUnit::RegisterType table, int address, int size)
{
	for (int i = 0; i < size; ++i)
	{
		quint16 regaddres = address + i;
		quint16 value = 0;
		quint16 last_value = 0;
		QString text = "";
		switch (table)
		{
		case (QModbusDataUnit::HoldingRegisters):
			mbServer->data(QModbusDataUnit::HoldingRegisters, regaddres, &value);
			lastServer->data(QModbusDataUnit::HoldingRegisters, regaddres, &last_value);

			if (last_value != value)
			{
				if (regaddres == modbus_map.value("HR_commands").toInt())
				{
					handler_set_commands(last_value, value);
				}
				if (regaddres == modbus_map.value("HR_FWD_power_setpoint").toInt())
				{
					handler_set_power_watt(value);
				}
				if (regaddres == modbus_map.value("HR_frequency_setpoint").toInt())
				{
					handler_set_frequency(value);
				}
				if (regaddres == modbus_map.value("HR_sweep_start_freq").toInt())
				{
					handler_set_SWP_frequency_start(value);
				}
				if (regaddres == modbus_map.value("HR_sweep_stop_freq").toInt())
				{
					handler_set_SWP_frequency_stop(value);
				}
				if (regaddres == modbus_map.value("HR_sweep_step_freq").toInt())
				{
					handler_set_SWP_frequency_step(value);
				}
				if (regaddres == modbus_map.value("HR_sweep_power").toInt())
				{
					handler_set_SWP_power(value);
				}
				if (regaddres == modbus_map.value("HR_DLL_frequency_lower").toInt())
				{
					handler_set_DLL_frequency_limit_lower(value);
				}
				if (regaddres == modbus_map.value("HR_DLL_frequency_upper").toInt())
				{
					handler_set_DLL_frequency_limit_upper(value);
				}
				if (regaddres == modbus_map.value("HR_DLL_frequency_start").toInt())
				{
					handler_set_DLL_frequency_start(value);
				}
				if (regaddres == modbus_map.value("HR_DLL_frequency_step").toInt())
				{
					handler_set_DLL_frequency_step(value);
				}
				if (regaddres == modbus_map.value("HR_DLL_threshold").toInt())
				{
					handler_set_DLL_threshold(value);
				}
				if (regaddres == modbus_map.value("HR_DLL_delay").toInt())
				{
					handler_set_DLL_delay(value);
				}
				if (regaddres == modbus_map.value("HR_PWM_frequency").toInt())
				{
					handler_set_PWM_frequency(value);
				}
				if (regaddres == modbus_map.value("HR_PWM_duty_cycle").toInt())
				{
					handler_set_PWM_duty_cycle(value);
				}
				if (regaddres == modbus_map.value("HR_PID_Kp").toInt())
				{
					handler_set_PID_Kp(value);
				}
				if (regaddres == modbus_map.value("HR_PID_Ki").toInt())
				{
					handler_set_PID_Ki(value);
				}
				if (regaddres == modbus_map.value("HR_PID_Kd").toInt())
				{
					handler_set_PID_Kd(value);
				}
				if (regaddres == modbus_map.value("HR_PID_setpoint").toInt())
				{
					handler_set_PID_setpoint(value);
				}
				if (regaddres == modbus_map.value("HR_PID_scaling_factor").toInt())
				{
					handler_set_PID_scaling_factor(value);
				}
				if (regaddres == modbus_map.value("HR_PID_offset").toInt())
				{
					handler_set_PID_offset(value);
				}
                if (regaddres == modbus_map.value("HR_phase").toInt())
                {
                    handler_set_phase(value);
                }
                if (regaddres == modbus_map.value("HR_clock_source").toInt())
                {
                    handler_set_clock_source(value);
                }
				if (regaddres == modbus_map.value("HR_PGB_ch1_phase").toInt())
				{
					handler_set_PGB_phase(1, value);
				}
				if (regaddres == modbus_map.value("HR_PGB_ch2_phase").toInt())
				{
					handler_set_PGB_phase(2, value);
				}
				if (regaddres == modbus_map.value("HR_PGB_ch3_phase").toInt())
				{
					handler_set_PGB_phase(3, value);
				}
				if (regaddres == modbus_map.value("HR_PGB_ch4_phase").toInt())
				{
					handler_set_PGB_phase(4, value);
				}
			}

			lastServer->setData(QModbusDataUnit::HoldingRegisters, regaddres, value);
			break;
		default:
			break;
		}
	}
}

void ModbusServer::handler_set_commands(quint16 lastVal, quint16 newVal)
{
	for (int i = 0; i < 16; i++)
	{
		if ((lastVal & (1<<i)) != (newVal & (1<<i)))
		{
			switch(i)
			{
			case 0:			//RF enable
				emit signal_set_RF_enable(1, newVal & (1<<i));
				break;
			case 1:			//Reset
				if (newVal & (1<<i))
				{
					// Only reset if the new value on the relevant bit is 'true';
					qDebug() << "modbus -> Reset SGx board";
					emit signal_execute_reset_SGx(1);
				}
				break;
			case 2:			//Clear Error
				if (newVal & (1<<i))
				{
					qDebug() << "modbus -> Clear Error";
					emit signal_execute_error_clear(1);
				}
				break;
			case 3:			//Execute Sweep
				if (newVal & (1<<i))
				{
					/* It's plausible that the modbus device writes both sweep parameters and a sweep execute.
					 * To ensure the sweep is executed with the most up to date values provided by the modbus device they are fetched here again. */

					quint16 value = 0;
					mbServer->data(QModbusDataUnit::HoldingRegisters, modbus_map.value("HR_sweep_start_freq").toInt(), &value);
					double start_freq = (double)value * multiplier_map.value("HR_sweep_start_freq");

					mbServer->data(QModbusDataUnit::HoldingRegisters, modbus_map.value("HR_sweep_stop_freq").toInt(), &value);
					double stop_freq = (double)value * multiplier_map.value("HR_sweep_stop_freq");

					mbServer->data(QModbusDataUnit::HoldingRegisters, modbus_map.value("HR_sweep_step_freq").toInt(), &value);
					double step_freq = (double)value * multiplier_map.value("HR_sweep_step_freq");

					mbServer->data(QModbusDataUnit::HoldingRegisters, modbus_map.value("HR_sweep_power").toInt(), &value);
					double pow_watt = (double)value * multiplier_map.value("HR_sweep_power");

					emit signal_execute_sweep(1, start_freq, stop_freq, step_freq, convert_watt_to_dbm(pow_watt));
				}
				break;
			case 4:			//DLL_enable
				emit signal_set_DLL_enable(1, newVal & (1<<i));
				break;
			case 5:			//PWM enable
				emit signal_set_PWM_enable(1, newVal & (1<<i));
				break;
			case 6:			//PID enable
//				emit signal_set_PID_enable(1, newVal & (1<<i));
				break;
			case 7:
				//Datalogging enable no longer used; is auto-enable when PSU's are enabled.
//				emit signal_set_datalogging_enable(newVal & (1<<i));
				if (newVal & (1<<i))
				{
					qDebug() << "modbus -> Power Control Mode: Feed Forward";
					emit signal_set_power_control_mode(1, POWER_CONTROL_FEED_FWD);
				}
				else
				{
					qDebug() << "modbus -> Power Control Mode: Autogain";
					emit signal_set_power_control_mode(1, POWER_CONTROL_NORMAL);
				}
				break;
			case 8:
				if (newVal & (1<<i))
				{
					qDebug() << "modbus -> Reset Protection Board";
					emit signal_execute_reset_protection(1);
				}
				break;
			case 9:
				emit signal_PSU_interlock(newVal & (1<<i));
				break;
			case 10:
				if (newVal & (1<<i))
				{
					emit signal_execute_restart_program();
				}
				break;
			case 15:		//TX changed
				setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 15, (newVal & (1<<i)));
				break;
			default:
				break;
			}
		}
	}
}

void ModbusServer::handler_set_power_watt(quint16 value)
{
	double power = (double)value * multiplier_map.value("HR_FWD_power_setpoint");
	emit signal_set_power_watt(1, power);
}

void ModbusServer::handler_set_frequency(quint16 value)
{
	double frequency = (double)value * multiplier_map.value("HR_frequency_setpoint");
	emit signal_set_frequency(1, frequency);
}

void ModbusServer::handler_set_SWP_frequency_start(quint16 value)
{
	double frequency = (double)value * multiplier_map.value("HR_sweep_start_freq");
	emit signal_set_SWP_frequency_start(1, frequency);
}

void ModbusServer::handler_set_SWP_frequency_stop(quint16 value)
{
	double frequency = (double)value * multiplier_map.value("HR_sweep_stop_freq");
	emit signal_set_SWP_frequency_stop(1, frequency);
}

void ModbusServer::handler_set_SWP_frequency_step(quint16 value)
{
	double frequency = (double)value * multiplier_map.value("HR_sweep_stop_freq");
	emit signal_set_SWP_frequency_step(1, frequency);
}

void ModbusServer::handler_set_SWP_power(quint16 value)
{
	double power = (double)value * multiplier_map.value("HR_sweep_power");
	emit signal_set_SWP_power(1, power);
}

void ModbusServer::handler_set_DLL_frequency_limit_lower(quint16 value)
{
	double frequency = (double)value * multiplier_map.value("HR_DLL_frequency_lower");
	emit signal_set_DLL_frequency_limit_lower(1, frequency);
}

void ModbusServer::handler_set_DLL_frequency_limit_upper(quint16 value)
{
	double frequency = (double)value * multiplier_map.value("HR_DLL_frequency_upper");
	emit signal_set_DLL_frequency_limit_upper(1, frequency);
}

void ModbusServer::handler_set_DLL_frequency_start(quint16 value)
{
	double frequency = (double)value * multiplier_map.value("HR_DLL_frequency_start");
	emit signal_set_DLL_frequency_start(1, frequency);
}

void ModbusServer::handler_set_DLL_frequency_step(quint16 value)
{
	double frequency = (double)value * multiplier_map.value("HR_DLL_frequency_step");
	emit signal_set_DLL_frequency_step(1, frequency);
}

void ModbusServer::handler_set_DLL_threshold(quint16 value)
{
	double threshold = (double)value * multiplier_map.value("HR_DLL_threshold");
	emit signal_set_DLL_threshold(1, threshold);
}

void ModbusServer::handler_set_DLL_delay(quint16 value)
{
	double delay = (double)value * multiplier_map.value("HR_DLL_delay");
	emit signal_set_DLL_delay(1, delay);
}

void ModbusServer::handler_set_PWM_frequency(quint16 value)
{
	double frequency = (double)value * multiplier_map.value("HR_PWM_frequency");
	emit signal_set_PWM_frequency(1,frequency);
}

void ModbusServer::handler_set_PWM_duty_cycle(quint16 value)	//Duty Cycle must be an int value between 1 and 99; multiplier not applicable.
{
	emit signal_set_PWM_duty_cycle(1,(int)value);
}

void ModbusServer::handler_set_PID_Kp(quint16 value)
{
	double kVal = (double) value * multiplier_map.value("HR_PID_Kp");
	emit signal_set_PID_Kp(1,(double)kVal);
}

void ModbusServer::handler_set_PID_Ki(quint16 value)
{
	double kVal = (double) value * multiplier_map.value("HR_PID_Ki");
	emit signal_set_PID_Ki(1,kVal);
}

void ModbusServer::handler_set_PID_Kd(quint16 value)
{
	double kVal = (double) value * multiplier_map.value("HR_PID_Kd");
	emit signal_set_PID_Kd(1,kVal);
}

void ModbusServer::handler_set_PID_setpoint(quint16 value)
{
	double setpoint = (double) value * multiplier_map.value("HR_PID_setpoint");
	emit signal_set_PID_setpoint(1,setpoint);
}

void ModbusServer::handler_set_PID_scaling_factor(quint16 value)
{
	double scale = (double) value * multiplier_map.value("HR_PID_scaling_factor");
	emit signal_set_PID_scaling(1, scale);
}

void ModbusServer::handler_set_PID_offset(quint16 value)
{
	double offset = (double) value * multiplier_map.value("HR_PID_offset");
	emit signal_set_PID_offset(1,offset);
}

void ModbusServer::handler_set_phase(quint16 value)
{
    double phase = (double) value * multiplier_map.value("HR_phase");
    emit signal_set_phase(1, phase);
}

void ModbusServer::handler_set_clock_source(quint16 value)
{
    if (value < 4)
	{
        emit signal_set_clock_source(1, value);
	}
}

/* Set Phase values from different channels of a Phase Gain Board */
void ModbusServer::handler_set_PGB_phase(quint16 PGB_channel_num, quint16 value)
{
	double PGB_phase = 0.0;
	switch (PGB_channel_num)
	{
	case 1:
			PGB_phase = (double) value * multiplier_map.value("HR_PGB_ch1_phase");
			break;
	case 2:
			PGB_phase = (double) value * multiplier_map.value("HR_PGB_ch2_phase");
			break;
	case 3:
			PGB_phase = (double) value * multiplier_map.value("HR_PGB_ch3_phase");
			break;
	case 4:
			PGB_phase = (double) value * multiplier_map.value("HR_PGB_ch4_phase");
			break;
	default:
		break;
	}

	emit signal_set_PGB_phase(1, PGB_channel_num, PGB_phase);
}
